hwex.sim <- function(allele1,allele2,female=NULL,n.sim=1000,iseed=NULL){

# Title: HWE test by simulations (approximates exact test for multiple alleles)

# Simulated  pval for approximating the exact test for HWE with
# multiple alleles (see Weir, Genetic Data Analysis II, page 108)
# input allele1 and allele2 are vectors of alleles with length = n,
# n = number of subjects
#
# The parameter iseed allows the user to specify a seed to be used to
# generate random numbers (useful for reproducing results).  If no seed is 
# specified, random variables are generated with the current value of 
# .Random.seed.


# allele1 and allele2 are vectors of alleles w/o missing values

if(!is.null(female))                    # Males possible
  obs <- hwex.stats(allele1, allele2, female)
else                                    # Assume all female
  obs <- hwex.stats(allele1, allele2, female=TRUE)

ret.code <- numeric(1)
ret.list <- numeric(3)

if(!is.null(iseed))
  { set.seed(iseed) }

seed.array <- runif(3)

# The seeds for ranAS183 must be between 1 and 30000, but bigger is 
# better (we think), so we add 10000

iseed1 = 10000 + 20000*seed.array[1]
iseed2 = 10000 + 20000*seed.array[2]
iseed3 = 10000 + 20000*seed.array[3]

if(!is.null(female))                    # Males possible
  { # Separate allele1 and allele2 into males and females:
     a1 <- allele1[female]
     a2 <- allele2[female]
     a3 <- allele1[!female]

     # Eliminate any NA values:
     na <- is.na(a1) | is.na(a2)
     a1 <- a1[!na]
     a2 <- a2[!na]
     a3 <- a3[!is.na(a3)]
   }
else                                    # Assume all female
  {  # Eliminate any NA values:
     na <- is.na(a1) | is.na(a2)
     a1 <- a1[!na]
     a2 <- a2[!na]
     a3 <- NULL
   }

out <- .C("hwex_sim",
   alleles=as.integer(c(a1,a2,a3)),
   n.sim=as.integer(n.sim),
   obs.nf=as.integer(obs$n.f),
   obs.nm=as.integer(obs$n.m),
   obs.nallele=as.integer(obs$n.allele),
   obs.exf=as.double(obs$ex.f),
   obs.exm=as.double(obs$ex.m),
   obs.u=as.double(obs$u.score),
   iseed1=as.integer(iseed1),
   iseed2=as.integer(iseed2),
   iseed3=as.integer(iseed3),
   rc=as.integer(ret.code),
   ret.list=as.double(ret.list),
   PACKAGE="hwe")

if(out$rc != 0)
  {
    if(out$rc == -1)
      warning("Number of simulations <= zero")
    if(out$rc == -2)
      warning("Length of allele <= zero: hwex.sim failed")
    if(out$rc == -3)
      warning("Number of female allele codes <= zero: hwex.sim failed")
    if(out$rc == -4)
      warning("Memory allocation failed: hwex.sim failed")
    if(out$rc == -5)
      warning("Number of male allele codes < zero: hwex.sim failed")
  }

pval.chi <- out$ret.list[1]
pval.like <- out$ret.list[2]
pval.rare <- out$ret.list[3]

return(list(pval.like=pval.like,pval.chi=pval.chi,pval.rare=pval.rare))

}






