# This function print.hwe takes an argument x that is a member of class "hwe"
# and prints a tablular format to standard output.  If the number of distinct
# allele codes in the locus object upon which Hardy-Weinberg Equilibrium is being
# performed is two, then the number of simulations requested is ignored and the
# output has the form
#
#                    stat       df          pval 
#        gof          val      val           val
#       rare          val        1           val
# like.exact            -        -           val
#
# with "val" denoting computed results and the row labeled "exact" indicating
# that Hardy-Weinberg Equilibrium simulations where not performed since the
# prescence of only two allele codes implies the corresponding presence of
# an easily computable exact method.  Hence, this method is thus used.  If, on
# the other hand, more than two allele codes are present and simulations are
# requested, then the output has the alternate form
#
#                    stat       df          pval 
#      gof            val      val           val
#     rare            val        1           val
#  gof.chi              -        -           val
# rare.sim              -        -           val
# like.sim              -        -           val
#
# with "val" denoting computed results and the three bottom rows representing
# the p-values obtained from Hardy-Weinberg Equilibrium simulations.  And in
# the case of there being more than two allele codes present and the number of
# simulations being less than or equal to zero, the output has the form
#
#                stat       df          pval 
#  gof            val      val           val
# rare            val        1           val
#
# where "val" denotes computational results and the last three rows are
# removed since there are no simulated values to output.


print.hwe <- function(x, digits=max(options()$digits-4,3), ...)
{

# Check whether we are using simulations or an exact test.

 if(is.null(x$like.exact.pval))
   {
 tbl <- matrix(NA,nrow=5,ncol=3)
 dimnames(tbl) <- list(c("gof     ","rare     ","gof.sim     ",
			 "rare.sim     ",       "like.sim     "),
                       c("stat","  df","  pval"))
 tbl[1,1] <- x$gof.stat
 tbl[1,2] <- x$gof.df
 tbl[1,3] <- x$gof.pval
 tbl[2,1] <- x$rare.stat
 tbl[2,2] <- 1 
 tbl[2,3] <- x$rare.pval 
 tbl[3,3] <- ifelse(is.null(x$gof.sim.pval), NA, x$gof.sim.pval)
 tbl[4,3] <- ifelse(is.null(x$rare.sim.pval), NA, x$rare.sim.pval)
 tbl[5,3] <- ifelse(is.null(x$like.sim.pval), NA, x$like.sim.pval)
}
 else
   {
 tbl <- matrix(NA,nrow=3,ncol=3)
 dimnames(tbl) <- list(c("gof     ","rare     ","like.exact     "),
                       c("stat","   df","  pval"))
 tbl[1,1] <- x$gof.stat
 tbl[1,2] <- x$gof.df
 tbl[1,3] <- x$gof.pval
 tbl[2,1] <- x$rare.stat
 tbl[2,2] <- 1 
 tbl[2,3] <- x$rare.pval
 tbl[3,3] <- x$like.exact.pval
}
 
# If we have enough allele codes to do simulations, then we need to check to
# be sure that simulations were requested.

 if(is.null(x$like.exact.pval) && is.null(x$gof.sim.pval))
{
# Round to the number of digits requested and strip the last three simulated
# rows from the table because no simulations were requested.
 
 tbl <- round(tbl,digits)
 tbl <- tbl[1:2,]
 print(tbl, quote=FALSE, right=TRUE) 
}
 else
{
  
# Round to the number of digits requested and replace NA with "-".

 tbl <- round(tbl,digits)
 tbl[is.na(tbl)] <- "-"
 print(tbl, quote=FALSE, right=TRUE)
 if(is.null(x$like.exact.pval))
   { cat("\nThe number of simulations requested, n.sim, was",x$n.sim)
   }
}
     cat("\nx.linked=",x$x.linked,"\n",sep="")

 invisible()
 return(tbl)
}
