/*  SCCS     %W%    %G%     */
/*** This function computes the logarithm of the gamma function.
**** Algorithm AS 245 gives an accuracy of about
**** 10-12 significant decimal digits except for small regions around X = 1 and
**** X = 2, where the function goes to zero.
****
**** ALGORITHM AS245  APPL. STATIST. (1989) VOL. 38, NO. 2
**** 
**** Converted from FORTRAN to C.
***/    

#include <math.h>

double alngam(double xvalue,int *ifault) {

      double x, x1, x2, y, lngam;
/*
**     Coefficients of rational functions
*/
     double R1[9] = {
          -2.66685511495, -24.4387534237,
          -21.9698958928,  11.1667541262,
     	   3.13060547623, 0.607771387771,
     	   11.9400905721,  31.4690115749,
     	   15.2346874070 };

     double R2[9] = {
           -78.3359299449, -142.046296688,
            137.519416416,  78.6994924154,
            4.16438922228,  47.0668766060,
            313.399215894,  263.505074721,
            43.3400022514 };

     double R3[9] = {
           -212159.572323,  230661.510616,
            27464.7644705, -40262.1119975,
           -2296.60729780, -116328.495004,
           -146025.937511, -24235.7409629,
           -570.691009324 };

     double R4[5] = {
           0.279195317918525,  0.4917317610505968,
           0.0692910599291889, 3.350343815022304,
           6.012459259764103 };

/*
**    Fixed constants
*/

#define ALR2PI 0.918938533204673
#define FOUR   4.0
#define HALF   0.5
#define ONE    1.0
#define ONEP5  1.5
#define TWELVE 12.0
#define ZERO   0.0

/*
**     Machine-dependant constants.
**     A table of values is given at the top of page 399 of the paper.
**     These values are for the IEEE double-precision format for which
**     B = 2, t = 53 and U = 1023 in the notation of the paper.
*/
#define XLGE  5.10e6 
#define XLGST 1.0e305

      x = xvalue;
      lngam = ZERO;
/*
**    Test for valid function argument
*/
      *ifault = 2;
      if (x >= XLGST) return lngam;
      *ifault = 1;
      if (x <= ZERO) return lngam;
      *ifault = 0;
/*
*     Calculation for 0 < X < 0.5 and 0.5 <= X < 1.5 combined
*/
      if (x < ONEP5) { 
	if (x < HALF) { 
	  lngam = -log(x);
	  y = x + ONE;
/*
*     Test whether X < machine epsilon
*/
	  if (y == ONE) return lngam;
        }
	else {
	  lngam = ZERO;
	  y = x;
	  x = (x - HALF) - HALF;
	}
	lngam = lngam + x * ((((R1[4]*y + R1[3])*y + R1[2])*y
                     + R1[1])*y + R1[0]) / ((((y + R1[8])*y + R1[7])*y
                     + R1[6])*y + R1[5]);
	return lngam;
      }
/*
*     Calculation for 1.5 <= X < 4.0
*/
      if (x < FOUR) {
	y = (x - ONE) - ONE;
	lngam = y * ((((R2[4]*x + R2[3])*x + R2[2])*x + R2[1])*x
                   + R2[0]) / ((((x + R2[8])*x + R2[7])*x + R2[6])*x
                   + R2[5]);
	return lngam;
      }
/*
*     Calculation for 4.0 <= X < 12.0
*/
      if (x < TWELVE) {
	lngam = ((((R3[4]*x + R3[3])*x + R3[2])*x + R3[1])*x + R3[0]) /
                 ((((x + R3[8])*x + R3[7])*x + R3[6])*x + R3[5]);
	return lngam;
      }
/*
*     Calculation for X >= 12.0
*/
      y = log(x);
      lngam = x * (y - ONE) - HALF * y + ALR2PI;
      if (x > XLGE) return lngam;
      x1 = ONE / x;
      x2 = x1 * x1;
      lngam = lngam + x1 * ((R4[2]*x2 + R4[1])*x2 + R4[0]) /
                   ((x2 + R4[4])*x2 + R4[3]);
      return lngam;
    }








