// $Author: sinnwell $ 
// $Date: 2010/05/27 20:07:50 $ 
// $Header: /people/biostat3/sinnwell/genet/rpack/Armitage/RCS/Matrix.hpp,v 1.2 2010/05/27 20:07:50 sinnwell Exp $ 
// $Locker:  $ 
// $Log: Matrix.hpp,v $
// Revision 1.2  2010/05/27 20:07:50  sinnwell
// put cpp code in here, so R package understand
//
// Revision 1.1  2007/10/15 19:34:54  sinnwell
// Initial revision
//


// To properly link template classes, need to include this Matrix.hpp
// header file in the file that uses the classes, and note that the 
// methods for the Matrix template classes are pulled in by the 
// include statement at the bottom of this file. Note that the Matrix.cpp
// file should then not be compiled separate from this mechanism.
// This is needed for Gnu g++.


#ifndef MATRIX_HPP
#define MATRIX_HPP


#include <stdio.h>
#include <stdlib.h>
#include <R.h>

// use C++ output for debugging on unix

#include <iostream>
#include <iomanip>

using std::setw;
using std::cout;
using std::endl;

template <class T>
class Matrix
{

public:

  Matrix (unsigned int r, unsigned int c, T *vec, int byCol);
  Matrix (unsigned int r, unsigned int c, T value);
  ~Matrix();

 
  int get_nrow() const;
  int get_ncol() const;

  void setElem(const unsigned int i, const unsigned int j, T val);
  void setAllElem(T val);

  T  operator() (unsigned int r, unsigned int c) const; // to ref as x(i,j)
  T& operator() (unsigned int r, unsigned int c);       // to ref as x(i,j) 
  T* operator[] (const int i);                          // to ref as x[i][j]        

  void print(const char * title) const;
  void print(const char * title, int maxDimPrint) const;

 protected:

  void asMatrixByRow( T *vec); 
  void asMatrixByCol( T *vec);
  void allocMatrixMemory();

  void errmsg(const char *msg) const;

  unsigned int  nrow;
  unsigned int  ncol;
  T **x;

};


template <class T>
class Table : public Matrix<T>
{

public:

  Table(unsigned int r, unsigned int c, T * vec, int byCol);
  Table(unsigned int r, unsigned int c, T value);
  ~Table();

  T get_rowTotal(int i) const;
  T get_colTotal(int i) const;
  T get_grandTotal(void) const;
 
  void computeMarginalTotals(void);  

  void print(const char * title) const;
  void print(const char * title, int maxDimPrint) const;

protected:

  T *rowTotal;
  T *colTotal;
  T grandTotal;

};



// Constructors

template <class T> 
Matrix<T> :: Matrix (unsigned int r, unsigned int c, T *vec, int byCol)
{
 

  nrow = r;
  ncol = c;

  allocMatrixMemory();

  if(byCol)
    {
      asMatrixByCol(vec);
    } else
    {
      asMatrixByRow(vec);
    }

}


template <class T> 
Matrix<T> :: Matrix (unsigned int r, unsigned int c, T value)
{
 
  unsigned int i, j;

  nrow = r;
  ncol = c;

  allocMatrixMemory();
 

  for(i=0; i < nrow; i++){
    for(j = 0; j < ncol; j++){
      x[i][j] = value;
    }
  }
  
}


// Destructor

template <class T> 
Matrix<T>::~Matrix(){

  delete [] x[0];
  delete [] x;

  return;
}


// get functions

template <class T> 
int Matrix<T>::get_nrow() const
{
  return nrow;
}


template <class T> 
int Matrix<T>::get_ncol() const
{
  return ncol;
}


// set functions

template <class T> 
void Matrix<T>::setElem(const unsigned int i,
			const unsigned int j,
			 T val){
  x[i][j] = val;
}

template <class T> 
void Matrix<T>::setAllElem(T val){
  unsigned int i,j;
  for(i=0; i<nrow; i++){
    for(j=0; j<ncol; j++){
      x[i][j] = val;
    }
  }

}


// operator overload functions


template <class T> 
T Matrix<T>::operator() (unsigned int r, unsigned int c) const
{
  if(r<0 | r >= nrow | c < 0 | c >= ncol)
    {
      errmsg("matrix indices out of bounds\n");
    }

  return (x[r][c]);
}

template <class T> 
T& Matrix<T> :: operator() (unsigned int r, unsigned int c)
{

  if(r<0 | r >= nrow | c < 0 | c >= ncol)
    {
      errmsg("matrix indices out of bounds\n");
    }

  return (x[r][c]);
}


template <class T>
T* Matrix<T> :: operator[] (const int i)
{
  return x[i];
}


// print functions (for debugging)


template <class T> 
void Matrix<T>::print(const char * title) const {

  unsigned int i, j;
 
  printf("%s\n", title);
  printf("nrow = %d, ncol = %d\n", nrow, ncol);


  for(i = 0; i < nrow; i++){
    for(j = 0; j < ncol; j++){
      // printf("%f ", x[i][j]);
      cout << x[i][j] << " ";
    }
    printf("\n");
  }


}

template <class T> 
void Matrix<T>::print(const char * title, int maxDimPrint) const 
{

  unsigned int i,j;
  unsigned int rowStop, colStop;
  printf("\n%s: ", title);
  printf(" (nrow = %d, ncol = %d)\n", nrow, ncol);

  rowStop = nrow > maxDimPrint ? maxDimPrint :  nrow;
  colStop = ncol > maxDimPrint ? maxDimPrint :  ncol;  
 

  for(i = 0; i < rowStop; i++){
    for(j = 0; j < colStop; j++){
      //      printf("%f ", x[i][j] );
      cout << x[i][j] << " ";
    }

    if(colStop < ncol){
      printf(" ...");
    }

    printf("\n");
  }
  
  if(rowStop < nrow){
    printf(".\n");
    printf(".\n");
    printf(".\n");
  }

}


// auxilliary functions

template <class T> 
void  Matrix<T> :: asMatrixByRow( T *vec)
{
 
  unsigned int i, j, k;

  k = 0;
  for(i=0; i < nrow; i++){
    for(j = 0; j < ncol; j++){
      x[i][j] = vec[k];
      k++;
    }
  }
}

template <class T> 
void Matrix<T> :: asMatrixByCol( T *vec)
{
 
  unsigned int i, j, k;
 
  k = 0;
  for(j = 0; j < ncol; j++){
      for(i=0; i < nrow; i++){
      x[i][j] = vec[k];
      k++;
    }
  }
}
template <class T> 
void Matrix<T>:: allocMatrixMemory() {

  unsigned int i;

  x = new (T * [nrow]);

 if (!x){
   errmsg("Error: Allocation failure 1 in allocMatrixMemory\n");
  }

 x[0] = new T [nrow * ncol];

  if (!x){
    errmsg("Error: Allocation failure 2 in allocMatrixMemory\n");
  }


  for(i= 1;i< nrow;i++) x[i] = x[i-1] + ncol;

  return;
}


template <class T> 
void Matrix<T> :: errmsg(const char * msg) const
{
  /* Function to emulate "stop" of S+ - see page 134, S Programing, by
     Venables and Ripley */

   PROBLEM "%s", msg RECOVER(NULL_ENTRY);
}



// Table class inherits from Matrix class


// constructors

template <class T>
Table<T> :: Table(unsigned int r, unsigned int c, T *vec, int byCol) :
  Matrix<T>(r, c, vec, byCol)
{

  // note need of this to access items in base class
  
  rowTotal = new T [this->nrow];
  colTotal = new T [this->ncol];

  computeMarginalTotals();
}

template <class T>
Table<T> :: Table(unsigned int r, unsigned int c, T value) :
  Matrix<T>(r, c, value)
{

  rowTotal = new T [this->nrow];
  colTotal = new T [this->ncol];

  computeMarginalTotals();
}

// destructor


template <class T>
Table<T> :: ~Table(){

  delete [] rowTotal;
  delete [] colTotal;

  return;
}

// get functions


template <class T>
T Table<T>:: get_rowTotal(int i) const
{
  if(i <0 | i >= (this->nrow))
    {
      this->errmsg("error in rowTotal index\n");
    }

  return rowTotal[i];
}

template <class T>
T Table<T>:: get_colTotal(int i) const
{
  if(i <0 | i >= (this->ncol) )
    {
      this->errmsg("error in colTotal index\n");
    }

  return colTotal[i];
}

template <class T>
T Table<T> :: get_grandTotal(void) const
{
  return grandTotal;
}


// additional functions

template <class T>
void Table<T>:: computeMarginalTotals(void){
  unsigned int i, j;

  for(i=0; i<(this->nrow); i++)
    {
      rowTotal[i] = 0;
      for(j=0; j<(this->ncol); j++)
	{
	  rowTotal[i] += (this->x[i][j]);
	}
    }

  for(j=0; j<(this->ncol); j++)
    {
      colTotal[j] = 0;
      for(i=0; i<this->nrow; i++)
	{
	  colTotal[j] += (this->x[i][j]);
	}
    }

  grandTotal = 0;

  for(i=0; i<(this->nrow); i++){
    grandTotal += rowTotal[i];
  }

}



// print functions


template <class T>
void Table<T>::print(const char * title) const {

  unsigned int i, j;
 
  printf("%s\n", title);
  printf("nrow = %d, ncol = %d\n", this->nrow, this->ncol);

  for(j = 0; j <= (this->ncol+1); j++){
    printf("--------");
  }
  printf("\n");


  for(i = 0; i < this->nrow; i++)
    {
      //      printf("| ");
      cout << "| ";
	for(j = 0; j < this->ncol; j++)
	  {
	    // printf("%6d | ", x[i][j]);
	    cout.width(10);
	    cout << this->x[i][j] << " | ";
	  }
	//      printf(" %6d\n", rowTotal[i]);
	cout.width(10);
	cout << rowTotal[i] << " " << endl;
    }


  for(j = 0; j <= (this->ncol+1); j++){
    // printf("--------");
    cout << "--------";

  }
  cout << endl;

  //  printf("\n");


  //  printf("| ");
  cout << "| ";

  for(j = 0; j < this->ncol; j++){
    //    printf("%6d | ", colTotal[j]);
    cout.width(10);
    cout << colTotal[j] << " |";
  }
  //  printf(" %6d\n", grandTotal);
  cout.width(10);
  cout << grandTotal << endl << endl;

}



template <class T> 
void Table<T>::print(const char * title, int maxDimPrint) const 
{

  unsigned int i,j;
  unsigned int rowStop, colStop;

   cout.precision(3);
   cout.width(10);   
 
  printf("\n%s: ", title);
  printf(" (nrow = %d, ncol = %d)\n", this->nrow, this->ncol);


  rowStop = this->nrow > maxDimPrint ? maxDimPrint :  this->nrow;
  colStop = this->ncol > maxDimPrint ? maxDimPrint :  this->ncol;  
 

  for(i = 0; i < rowStop; i++){
    for(j = 0; j < colStop; j++){
      //      printf("%f ", x[i][j] );
      cout.width(10);   
      cout << this->x[i][j] << " ";
    }

    if(colStop < this->ncol){
      cout << (" ...");
    }
    cout.width(10);
    cout << " | " << rowTotal[i] << " " << endl;
    //printf("\n");
  }
  
  if(rowStop < this->nrow){
    printf(".\n");
    printf(".\n");
    printf(".\n");
  }


 for(j = 0; j < colStop; j++){
    //    printf("%6d | ", colTotal[j]);
   cout.width(10);
 
   cout << colTotal[j] << " ";
  }
  //  printf(" %6d\n", grandTotal);
 cout << (" ...");
 cout.width(10);
 cout << grandTotal << endl << endl;


}


#endif

