// $Author: sinnwell $ 
// $Date: 2007/10/15 19:34:08 $ 
// $Header: /people/biostat3/sinnwell/genet/rpack/Armitage/RCS/armitage.cpp,v 1.1 2007/10/15 19:34:08 sinnwell Exp $ 
// $Locker:  $ 
// $Log: armitage.cpp,v $
// Revision 1.1  2007/10/15 19:34:08  sinnwell
// Initial revision
//


#include "Matrix.hpp"

extern "C" {

#include <S.h> 
#include <limits.h>
#include "quicksortFolie.h"
#include "mt19937ar-cok.h"


int * intVector (unsigned int n);

double * doubleVector (unsigned int n);

void computeArmitageTable(Table<int> & armTable, const int * status,
			    const Matrix<int> & geno, unsigned int col);

double computeArmitageStat(Table<int> & armTable, const int * dose);

static void errmsg(const char *msg);


void armitage(
	      int *nsubj,
	      int *ngeno,
	      int *status,
	      int *genovec,
	      int *dose,
	      int *seed,
	      int *nsim,
	      double *statObs,
	      double *statSim)
{

  int byCol = 1; // to copy genovec into geno matrix by columns

  unsigned int i, j, isim;

  // Allocate Memory - be sure to Free at end

  Matrix<int> geno(*nsubj, *ngeno, genovec, byCol);
  
  // Create status (2 rows) by geno (3 cols) genotype
  // count table, and initialize to 0.

  Table<int> armTable(2,3,0);

  double *randomVec = doubleVector(*nsubj);
  
  // check limits:

  if(*nsim > INT_MAX)
    {
      errmsg("Error: nsim > INT_MAX");
    }



  // Begin Calculations
  
  for(j=0; j < *ngeno; j++)
    {
      computeArmitageTable(armTable, status, geno, j);
      statObs[j] = computeArmitageStat(armTable, dose);
    }



  if(*nsim > 0)
    {


      init_genrand(*seed);	

      unsigned int k = 0;
      
      for(isim = 0; isim < *nsim; isim++)
	{

	  for(i = 0; i < *nsubj; i++)
	    {
	      randomVec[i] = genrand_real1();
  
	    }
	  
	 
	  quicksort_permute(*nsubj, randomVec, status);


	  for(j=0; j < *ngeno; j++)
	    {
	      computeArmitageTable(armTable, status, geno, j);
	      statSim[k] = computeArmitageStat(armTable, dose);
	      k++;
	    }
	}
    }


  // Free Memory

  //  geno.freeMatrix();
  // armTable.freeTable();
  Free(randomVec);
 
}


// input a 2 x 3 contingency table and dose vec of length 3 (for scoring
// genotype dosages) and return Armitage trend chi-square statistic 
// (1 degree of freedom)

double computeArmitageStat(Table<int> & armTable, const int * dose){
  
  unsigned int i,j, nrow, ncol;
  
  nrow = armTable.get_nrow();
  ncol = armTable.get_ncol();


  int caseTot = 0;
  int controlTot = 0;
  int grandTot = 0;

  double fracCase;
  double fracControl;
  double u = 0;
  double sumsq = 0.0;
  double sum = 0.0;
  double vu;
  double stat;


  // for keeping track of cases and controls in armitage table:  
  //  CASE    ROW = 1
  //  CONTROL ROW = 0

  caseTot    = armTable.get_rowTotal(1);
  controlTot = armTable.get_rowTotal(0);
  grandTot   = armTable.get_grandTotal();


  if(grandTot == 0)
    {
      return 0.0;
    }

  fracCase    = double(caseTot) / double(grandTot);
  fracControl = 1.0 - fracCase;
 

  for(j=0; j<ncol; j++)
    {
      u += dose[j] * (fracControl * armTable(1,j) - fracCase * armTable(0,j));
    }


  for(j=0; j<ncol; j++)
    {
      sumsq += dose[j]*dose[j] * armTable.get_colTotal(j);
      sum += dose[j] * armTable.get_colTotal(j); 
    }

 
  vu =  ( (grandTot * sumsq - sum*sum) * fracCase * fracControl ) / double(grandTot);


  // check for vu approaching 0, set stat = 0 if true

  if(vu < 1e-6)
    {
      return 0.0;
    }


  stat =  u * u / vu;

  return stat;
}


// input an armTable reference that will hold a 2 x 3 contingency table, a
// status vector of length n (0 if case, 1 if control), a genotype matrix (n rows
// by G cols, where G = number of SNPs), with elements coded 0,1,2 (3 for missing values 
// which are ignored), and the col index of this matrix, to compute the armTable

void computeArmitageTable(Table<int> & armTable, const int * status,
			   const Matrix<int> & geno, unsigned int col){
  
   unsigned int i,j, nrow;
   int val = 0;

   armTable.setAllElem(0);

   nrow = geno.get_nrow();
 

   // Compute cell counts, for valid values 0, 1, 2 (skip missing values coded >= 3)

   for(i=0; i < nrow; i++){
     
     if(geno(i,col) < 3)
       {
	 armTable(status[i], geno(i,col)) ++;
       }
   }


   // Compute row, col, grand totals

   armTable.computeMarginalTotals();

   return;
}


// allocate space for a vector of n integers and return
// pointer to start of vector

int * intVector (unsigned int n)
{
  int * vec = (int *) Calloc(n, int);

  if (!vec){
    errmsg("Error: Allocation failure intVector\n");
  }

  return vec;
}



// allocate space for a vector of n doubles and return
// pointer to start of vector

double * doubleVector (unsigned int n)
{

  double * vec = (double *) Calloc(n, double);

  if (!vec){
    errmsg("Error: Allocation failure doubleVector\n");
  }

  return vec;
}



static void errmsg(const char *msg){

  /* Function to emulate "stop" of S+ - see page 134, S Programing, by
     Venables and Ripley */

   PROBLEM "%s", msg RECOVER(NULL_ENTRY);
}


}
